#include "CRfePurEngineeringDialog.h"

#include <QCloseEvent>

#include <License.h>
#include <permission/Permissions.h>

#include <reader/QrfeRfePurReader>

#include "CRfePurEngineeringDialog/CRfePurFrequencySweepMeasurement.h"
#include "CRfePurEngineeringDialog/CRfePurStaticReflectedPowerMeasurement.h"
#include "CRfePurEngineeringDialog/CRfePurNetworkAnalyzer.h"
#include "CRfePurEngineeringDialog/CRfePurAS399XRegisterTable.h"
#include "CRfePurEngineeringDialog/CRfePurInterfaceSettings.h"
#include "CRfePurEngineeringDialog/CRfePurReflectedPowerSettings.h"
#include "CRfePurEngineeringDialog/CRfePurHiddenSettings.h"
#include "CRfePurEngineeringDialog/CRfePurExperimentalSettings.h"
#include "CRfePurEngineeringDialog/CRfePurLabFunctions.h"

CRfePurEngineeringDialog::CRfePurEngineeringDialog(QWidget *parent)
    : QDialog(parent)
	, QrfeTraceModule("CRfePurEngineeringDialog")
	, m_reader(0)
	, m_normalEnginerring(false)
	, m_labFunctions(false)
	, m_hiddenSettings(false)
	, m_interfaceSettings(false)
    , m_reflectedPowerSettings(false)
	, m_advancedSettings(false)
	, m_originalAntenna(0)
	, m_originalAttenuation(0)
	, m_originalCyclicModus(false)
	, m_originalFrequencyMode(0)
{
	ui.setupUi(this);
	setInfoEdit(ui.infoEdit);

	this->setWindowFlags(this->windowFlags() | Qt::WindowMaximizeButtonHint);

	connect(ui.okButton, 					SIGNAL(clicked()),
			this, 							  SLOT(accept()));

	connect(ui.tabWidget, 					SIGNAL(currentChanged(int)),
			this, 							  SLOT(currentTabChanged(int)));

	connect(Permissions::d,					SIGNAL(permissionsChanged()),
			this, 							  SLOT(permissionsChanged()));

    //ui.licenseFrame->setVisible(!License::d->hasLicense(QrfeLicense::ReaderSuite_EngineeringLicense));
    ui.licenseFrame->setVisible(false);
}

CRfePurEngineeringDialog::~CRfePurEngineeringDialog()
{
	while(m_tabWidgets.size() > 0)
	{
		ReaderTabWidget* widget = m_tabWidgets.takeLast();
		delete widget;
	}
}

int CRfePurEngineeringDialog::exec(QrfeReaderInterface* reader)
{
	m_reader = qobject_cast<QrfeRfePurReader*>(reader);
	if( m_reader == 0 )
		return 0;

	ulong sw_rev = m_reader->softwareRevision();
	sw_rev &= 0x0000FFFF;
	if(sw_rev < 0x0027)
		return 0;

	clearInfo();

	ui.moduleLabel->setText(m_reader->readerModuleString());

	uchar maxFreq;
	m_reader->getCurrentAttenuation(m_originalAttenuation);
	m_reader->getFrequency(m_originalFrequencyMode, maxFreq, m_originalFrequency);

	ushort maxAtt = 0;
	if(m_reader->getMaxAttenuation(maxAtt) == QrfeGlobal::RES_OK)
		m_reader->setAttenuation(maxAtt);


	for(int i = 0; i < m_tabWidgets.size(); i++)
	{
		ReaderTabWidget* widget =  m_tabWidgets.at(i);
		if(!widget->init(m_reader))
			ui.tabWidget->setTabEnabled(i, false);
		else
			ui.tabWidget->setTabEnabled(i, true);
	}

	currentTabChanged(ui.tabWidget->currentIndex());

	connect(m_reader, 			SIGNAL(lostConnection()),
			this, 				  SLOT(reject()) );

	return QDialog::exec();
}

void CRfePurEngineeringDialog::done(int r)
{
	foreach(ReaderTabWidget* widget, m_tabWidgets)
	{
		widget->deinit();
	}

	m_reader->setAttenuation(m_originalAttenuation);
	m_reader->setFrequency(m_originalFrequencyMode, m_originalFrequency);
	m_reader->readSettings();

	disconnect( m_reader, 			SIGNAL(lostConnection()),
				this, 				  SLOT(reject()) );

	QDialog::done(r);
}

void CRfePurEngineeringDialog::tabWidgetWorking()
{
	ReaderTabWidget* w = qobject_cast<ReaderTabWidget*>(sender());
	if(w == 0)
		return;

	for(int i = 0; i < m_tabWidgets.size(); i++)
	{
		ReaderTabWidget* widget =  m_tabWidgets.at(i);
		if(widget == w)
			ui.tabWidget->setTabEnabled(i, true);
		else
			ui.tabWidget->setTabEnabled(i, false);
	}
	ui.okButton->setEnabled(false);
}

void CRfePurEngineeringDialog::tabWidgetIdle()
{
	for(int i = 0; i < m_tabWidgets.size(); i++)
	{
		ui.tabWidget->setTabEnabled(i, true);
	}
	ui.okButton->setEnabled(true);
	reloadExecutionCount(ui.tabWidget->currentIndex());
}

void CRfePurEngineeringDialog::closeEvent ( QCloseEvent * event )
{
	bool working = false;
	foreach(ReaderTabWidget* widget, m_tabWidgets)
	{
		if(widget->isWorking())
		{
			working = true;
			break;
		}
	}

	if(working){
		event->ignore();
		return;
	}

	QDialog::closeEvent(event);
}

void CRfePurEngineeringDialog::currentTabChanged( int index )
{
	foreach(ReaderTabWidget* widget, m_tabWidgets)
	{
		widget->clearView();
	}

	if(index >= m_tabWidgets.size() || index < 0)
		return;

	reloadExecutionCount(index);
	m_tabWidgets.at(index)->activated();
}

void CRfePurEngineeringDialog::addTabWidget(ReaderTabWidget* widget)
{
	m_tabWidgets << widget;
	ui.tabWidget->addTab(widget, widget->name());
	connect(widget, 	SIGNAL(working()),
			this, 		  SLOT(tabWidgetWorking()));
	connect(widget, 	SIGNAL(idle()),
			this, 		  SLOT(tabWidgetIdle()));
}

void CRfePurEngineeringDialog::reloadExecutionCount(int tabIndex)
{
	if(tabIndex >= m_tabWidgets.size() || tabIndex < 0)
		return;

	TabWidget* tab = m_tabWidgets.at(tabIndex);
	if(tab->hasExecutionCount())
	{
		ui.licenseFrame->setVisible(true);
		ui.executionCountLabel->setText(QString::number(tab->executionCount()));
	}
	else
		ui.licenseFrame->setVisible(false);
}

void CRfePurEngineeringDialog::permissionsChanged()
{
	if(Permissions::d->readerEngineering() && !m_normalEnginerring)
	{
		m_normalEnginerring = true;
		addTabWidget(new CRfePurFrequencySweepMeasurement(ui.infoEdit, this));
		addTabWidget(new CRfePurStaticReflectedPowerMeasurement(ui.infoEdit, this));
		addTabWidget(new CRfePurNetworkAnalyzer(ui.infoEdit, this));
	}
	if(Permissions::d->readerLabFunctions() && !m_labFunctions)
	{
		m_labFunctions = true;
		addTabWidget(new CRfePurLabFunctions(ui.infoEdit, this));
	}
	if(Permissions::d->readerHiddenSettings() && !m_hiddenSettings)
	{
		m_hiddenSettings = true;
		addTabWidget(new CRfePurHiddenSettings(ui.infoEdit, this));
	}
    if(Permissions::d->readerInterfaceSettings() && !m_interfaceSettings)
    {
        m_interfaceSettings = true;
        addTabWidget(new CRfePurInterfaceSettings(ui.infoEdit, this));
    }
    if(Permissions::d->readerReflectedPowerSettings() && !m_reflectedPowerSettings)
    {
        m_reflectedPowerSettings = true;
        addTabWidget(new CRfePurReflectedPowerSettings(ui.infoEdit, this));
    }
    if(Permissions::d->readerAdvancedSettings() && !m_advancedSettings)
	{
		m_advancedSettings = true;
		addTabWidget(new CRfePurExperimentalSettings(ui.infoEdit, this));
		addTabWidget(new CRfePurAS399XRegisterTable(ui.infoEdit, this));
	}
}
